/*
 * $Id: liman.h 286 2020-10-11 20:12:42Z baroks $
 *
 * $Log$
 *
 */

#ifndef __LIMAN_H__
#define __LIMAN_H__

#ifdef __cplusplus
extern "C" {
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#ifndef WIN32
#ifdef _LIMAN_DLL_
#undef _LIMAN_DLL_
#endif
#endif

#ifdef _LIMAN_DLL_  
#ifdef _LIMAN_DLL_EXPORTS_    
  /* building libliman_sdk.dll */
  #define STDCALL       __stdcall
  #define PRECALL       __declspec(dllexport)
#else
  /* calling libliman_sdk.dll */
  #define STDCALL       __stdcall
  #define PRECALL       
#endif
#else
  #define STDCALL
  #define PRECALL
#endif


typedef struct LIMENV     *pLIMENV;
typedef struct LIMLIC     *pLIMLIC;


/* version ranges */
#define LIMVER_MAJOR                            1    /*!< Major version of current LIMAN SDK distribution */   
#define LIMVER_MINOR                            2    /*!< Minor version of current LIMAN SDK distribution */   
#define LIMVER_RELEASE                          5    /*!< Release identifier of current LIMAN SDK distribution */   
#define LIMVER_REVISION                         285  /*!< Revision identifier of current LIMAN SDK distribution */

/* defaults */
#define LIMPAR_NONE                            -1
#define LIMPAR_DEFAULT                          0
#define LIMPAR_ENCRYPT                          0
#define LIMPAR_DECRYPT                          1
#define LIMPAR_PUBKEY                           1
#define LIMPAR_PRIVKEY                          2

/** @defgroup dfCONSTANTS LIMAN Symbolic Constants 
@{
*/

/**
 * \defgroup gHOST Host identifiers
 * \brief List of host identifiers to create locked licenses
 */
/**@{*/
#define LIMPAR_CPUID                            1   /*!< CPU identifier */   
#define LIMPAR_DISKID                           2   /*!< HardDisk identifier */   
#define LIMPAR_HOSTNAME                         4   /*!< Hostname */   
#define LIMPAR_USERID                           8   /*!< Username */   
#define LIMPAR_DMI                              16  /*!< DMI identifier (Linux only) */
#define LIMPAR_MACID                            32  /*!< MAC identifier (1st) */   
#define LIMPAR_MACID_2                          64  /*!< MAC identifier (2st) */   
#define LIMPAR_MACID_3                          128  /*!< MAC identifier (3rd) */   
#define LIMPAR_MACID_4                          256  /*!< MAC identifier (4rd) */   
/**@}*/

/**
 * \defgroup gMISCPAR  Miscellenous constants
 * \brief List of constants
 */
/**@{*/
#define LIMPAR_MAX_PATH                      2048   /*!< Max-allowable path length */   
#define LIMPAR_MAX_CHARS_PER_LINE             256   /*!< Max-allowable characters per line in text input files */   
#define LIMPAR_MAX_LIC_STRING_LEN            4096   /*!< Max characters of license string */   
#define LIMPAR_NUMBER_OF_ERROR_CODES           52   /*!< Max number of error codes, see LIMERR_LAST_CARD */   
#define LIMPAR_MAX_LENGTH_ERROR_MESSAGE        32   /*!< Max length of error messages */
#define LIMPAR_NUM_FEATURE_SLOTS                4   /*!< Max number of feature slots */
/**@}*/

/**
 * \defgroup gID_LEN String lengths
 * \brief List of string lengths for various data
 */
/**@{*/
#define LIMPAR_MAX_LEN_PRODUCTID               12    /*!< Max allowable length of product-id */   
#define LIMPAR_MAX_LEN_UUID                    16    /*!< The length of UUID */   
#define LIMPAR_MAX_LEN_SERIAL                  24    /*!< Max allowable length of serial string */   
#define LIMPAR_MAX_LEN_HOSTID                  12    /*!< The length of hostid string */   
#define LIMPAR_MAX_LEN_USERID                  12    /*!< Max allowable length of hostid string */   
#define LIMPAR_MAX_LEN_LICID                   16    /*!< The length of license-id string */   
/**@}*/

/**
 * \defgroup gERROR Error codes
 * \brief Error codes in LIMAN SDK. See \ref pidAPPENDIXA "Appendix A"
 * for descriptions.
 */
/**@{*/
#define LIM_OK                             0      /*!< No errors */   
#define LIMERR_FIRST_CARD                  1      /*!< Placeholder for first error */
#define LIMERR_MEMORY                      2      /*!< Error in allocating memory */
#define LIMERR_FILE_NOT_FOUND              3      /*!< Specified file was not found */
#define LIMERR_INVALID_NULL                4      /*!< Specified reference is invalid or null */
#define LIMERR_INVALID_LICENSE             5      /*!< Specified license is invalid */
#define LIMERR_ENVAR_NOT_FOUND             6      /*!< Specified environment variable is not found */
#define LIMERR_LICENSE_EXPIRED             7      /*!< License has expired */
#define LIMERR_LICENSE_NOT_LOADED          8      /*!< License is not loaded */
#define LIMERR_INVALID_DOMAIN              9      /*!< Domain parameters are invalid */
#define LIMERR_INVALID_PRODUCT_CODE        10      /*!< Specified product code is invalid for the given license */
#define LIMERR_INVALID_USER                11      /*!< Current user is invalid for the given license */
#define LIMERR_INVALID_PRODUCT_VERSION     12      /*!< Specified product version is invalid for the given license */
#define LIMERR_INVALID_PLATFORM            13      /*!< Specified platform code is invalid for the given license */
#define LIMERR_INVALID_ENVAR               14      /*!< Specified environment variable is invalid */
#define LIMERR_INVALID_ENV                 15      /*!< Specified environment is invalid */
#define LIMERR_INVALID_INPUT               16      /*!< Specified input is invalid */
#define LIMERR_UNABLE_TO_OPEN_FILE         17      /*!< Failed to open specified file */
#define LIMERR_INVALID_LICENSE_FILE_NAME   18      /*!< Specified license file is invalid */
#define LIMERR_OPERATION_FAILED            19      /*!< An internal crypto-operation has failed */
#define LIMERR_PRIVKEY_FILE_NOT_FOUND      20      /*!< Private key file is not found */
#define LIMERR_PUBKEY_FILE_NOT_FOUND       21      /*!< Public key file is not found */
#define LIMERR_NOT_SUPPORTED               22      /*!< Feature is not supported */
#define LIMERR_INVALID_KEYLEN              23      /*!< The key length is invalid */
#define LIMERR_NOT_AVAILABLE               24      /*!< Requested information is not available */
#define LIMERR_INVALID_KEY                 25      /*!< Key under operation is invalid */
#define LIMERR_SHORT_KEY                   26      /*!< Key under operation is too short */
#define LIMERR_BAD_ERRORCODE               27      /*!< Specified error code is invalid */
#define LIMERR_INTERNAL                    28      /*!< An internal error has occurred. Contact your vendor */
#define LIMERR_ENV_NOT_READY               29      /*!< LIMAN environment is not ready */
#define LIMERR_ALREADY_LOADED              30      /*!< Specified object is already loaded */
#define LIMERR_SYSTEM                      31      /*!< Platform specific system error */
#define LIMERR_CLOCK_SETBACK               32      /*!< Clock setback detected */
#define LIMERR_INVALID_HOST                33      /*!< Host is invalid or hardware has changed */
#define LIMERR_KEYS_NOT_LOADED             34      /*!< Private and Public keys are not loaded */
#define LIMERR_PRIVKEY_FILE_EXISTS         35      /*!< Private key file 'usr_private.key' already exists */
#define LIMERR_PUBKEY_FILE_EXISTS          36      /*!< Public key file 'usr_public.key' already exists */
#define LIMERR_KEYSIZE_MISMATCH_ENV        37      /*!< The size of specified key does not conform the environment */
#define LIMERR_RANGE_TYPE_CODE             38      /*!< Specified type-code is out-of-range */
#define LIMERR_RANGE_PRODUCT_CODE          39      /*!< Specified product-code is out-of-range */
#define LIMERR_RANGE_MAJOR_VERSION         40      /*!< Specified major-version is out-of-range */
#define LIMERR_RANGE_MINOR_VERSION         41      /*!< Specified minor-version is out-of-range */
#define LIMERR_RANGE_SIZE_CODE             42      /*!< Specified size code is out-of-range */
#define LIMERR_RANGE_PLATFORM_CODE         43      /*!< Specified platform code is out-of-range. */
#define LIMERR_INVALID_HOSTMASK            44      /*!< Host is invalid or hardware has changed */
#define LIMERR_BUFFER_TOO_LONG             45      /*!< Buffer is too long */
#define LIMERR_MKDIR_FAILED                46      /*!< Cannot create directory */
#define LIMERR_PRIVKEY_NOT_LOADED          47      /*!< Private key is not loaded */
#define LIMERR_PUBKEY_NOT_LOADED           48      /*!< Public key is not loaded */
#define LIMERR_FILE_TOO_SMALL              49      /*!< Specified file is too small */
#define LIMERR_FILE_IO                     50      /*!< File I/O error */
#define LIMERR_INPUT_TOO_BIG               51      /*!< Specified input is too big */
#define LIMERR_LAST_CARD                   52      /*!< Placeholder for last error */
/**@}*/

/**
 * \defgroup gLICSPEC License properties
 * \brief List of license properties
 */
/**@{*/
#define LIMPAR_PLATFORM                       101  /*!< License Platform identifier */
#define LIMPAR_CODE                           102  /*!< Reserved */
#define LIMPAR_MAJORVER                       103  /*!< License Major version */
#define LIMPAR_MINORVER                       104  /*!< License Minor version */
#define LIMPAR_SIZE                           105  /*!< License Size identifier */
#define LIMPAR_TYPE                           106  /*!< License Type identifier */
#define LIMPAR_SEATS                          107  /*!< License max-seats */
#define LIMPAR_EXPIRY                         108  /*!< License expiry date in YYYYMMDD format */
#define LIMPAR_RESERVED                       109  /*!< Reserved */
#define LIMPAR_FEATURE0                       110  /*!< License feature-0 bitmask  */
#define LIMPAR_FEATURE1                       111  /*!< License feature-1 bitmask  */
#define LIMPAR_FEATURE2                       112  /*!< License feature-2 bitmask  */
#define LIMPAR_FEATURE3                       113  /*!< License feature-3 bitmask  */
#define LIMPAR_SERIAL                         114  /*!< License serial string  */
#define LIMPAR_USERINFO                       115  /*!< License user identifier  */
#define LIMPAR_HOSTID                         116  /*!< License hostid  */
/**@}*/


/**        
 *  \defgroup gSAMPLE_MACROS Sample symbolic constants to set up a license
 *
 *  \brief This is a list of symbolic constants provided in \c include\liman.h 
 *  for grouping the input required to set up a license into sensible sets. The 
 *  symbols and their values can be modified as the user sees fit or they could 
 *  be ignored altogether because LIMAN SDK does not use them internally. These 
 *  constants are intended to be used mainly with lim_load_lic() or when checking
 *  license properties (lim_platform_id(), lim_product_id(), etc..)
 *
 */
/**@{*/

/**
 * \defgroup gVERSION Version constants
 * \brief Version tags can be integer values in 0-255 range
 */
/**@{*/
#define LIMPROD_MAJOR_VER                       1   /*!< Placeholder reserved for optional product version */
#define LIMPROD_MINOR_VER                       0   /*!< Placeholder reserved for optional product version */
/**@}*/

/**
 * \defgroup gPLATFORM Platform constants
 * \brief Platform identifiers can be integer values in 0-255 range
 */
/**@{*/
#define LIMPLAT_WIN32X86                        1   /*!< Identifies 32-bit Windows operating system on x86 architecture */
#define LIMPLAT_LNX32X86                        2   /*!< Identifies 32-bit Linux operating system on x86 architecture */
#define LIMPLAT_OSX32X86                        3   /*!< Identifies 32-bit OSX operating system on x86 architecture */
#define LIMPLAT_SOL32X86                        4   /*!< Identifies 32-bit Solaris operating system on x86 architecture */
#define LIMPLAT_SSPARC32                        5   /*!< Identifies 32-bit Solaris operating system on sparc architecture */
#define LIMPLAT_WIN64X86                        6   /*!< Identifies 64-bit Window operating system on x86_64 architecture */
#define LIMPLAT_LNX64X86                        7   /*!< Identifies 64-bit Linux operating system on x86_64 architecture */
#define LIMPLAT_OSX64X86                        8   /*!< Identifies 64-bit OSX operating system on x86_64 architecture */
#define LIMPLAT_SOL64X86                        9   /*!< Identifies 64-bit Solaris operating system on x86_64 architecture */
#define LIMPLAT_SSPARC64                       10   /*!< Identifies 64-bit Solaris operating system on sparc architecture */
#define LIMPLAT_MINGW32                        11   /*!< Identifies 32-bit MinGW toolchain for x86 architecture */
#define LIMPLAT_MINGW64                        12   /*!< Identifies 64-bit MinGW toolchain for x64 architecture */
/**@}*/


/**
 * \defgroup gSIZE License-size constants
 * \brief Size constant can be integer values in 0-255 range
 */
/**@{*/
#define LIMSIZE_DEMO                          101   /*!< Identifies a demo/trial license */
#define LIMSIZE_STANDARD                      102   /*!< Identifies a standard license */
#define LIMSIZE_EXTENDED                      103   /*!< Identifies a extended license */
/**@}*/

/**
 * \defgroup gTYPE License-type constants
 * \brief License type can be integer values in 0-255 range
 */
/**@{*/
#define LIMTYPE_EDUCATIONAL                   111   /*!< Identifies an educational license */
#define LIMTYPE_COMMERCIAL                    112   /*!< Identifies an commercial license */
#define LIMTYPE_DEVELOPMENT                   113   /*!< Identifies an development license */
/**@}*/

/**
 * \defgroup gFEATURES Feature bitmasks
 * \brief Each feature slot can take integer values in 0-255 range
 *  enabling/disabling a subset of 8 features.
 */
/**@{*/
#define LIMOPT_FEATURE_0                       1    /*!< Bitmask value associated with feature 0 (1'st bit) */
#define LIMOPT_FEATURE_1                       2    /*!< Bitmask value associated with feature 1 (2'nd bit) */
#define LIMOPT_FEATURE_2                       4    /*!< Bitmask value associated with feature 2 (3'rd bit) */
#define LIMOPT_FEATURE_3                       8    /*!< Bitmask value associated with feature 3 (4'th bit) */
#define LIMOPT_FEATURE_4                      16    /*!< Bitmask value associated with feature 4 (5'th bit) */
#define LIMOPT_FEATURE_5                      32    /*!< Bitmask value associated with feature 5 (6'th bit) */
#define LIMOPT_FEATURE_6                      64    /*!< Bitmask value associated with feature 6 (7'th bit) */
#define LIMOPT_FEATURE_7                     128    /*!< Bitmask value associated with feature 7 (8'th bit) */
/**@}*/

/**@}*/ //end gSAMPLE_MACROS

/**@}*/ //end dfCONSTANTS

/* end of macro definition */


/*** function prototypes ***/

PRECALL
pLIMENV STDCALL lim_create_env(int nKeySize, int nHashSize, int *pnErr);

PRECALL
int STDCALL lim_free_lic(pLIMLIC *pLic);

PRECALL
int STDCALL lim_delete_lic(pLIMLIC pLic);

PRECALL
int STDCALL lim_free_env(pLIMENV *pEnv);

PRECALL
int STDCALL lim_delete_env(pLIMENV pEnv);

PRECALL
int STDCALL lim_load_lic(pLIMLIC pLic,
                             const char  *szProductCode,                             
                             int   mPlatformId,
                             int   nMajorVersion,
                             int   nMinorVersion,
                             int   mSizeId,
                             int   mTypeId,
                             int   nUsers,
                             int   nExpiration,
                             const char  *pcFeatures,
                             const char  *szSerial,
                             const char  *szUUID,
                             const char  *szHost,
                             const char  *szUserId,
                             int   nHostIdMask);

PRECALL 
int STDCALL lim_set_licreq(  pLIMLIC pLic,
                             int   mPlatformId,
                             char  *szProductCode,
                             int   nMajorVersion,
                             int   nMinorVersion,
                             int   mSizeId,
                             int   mTypeId);

PRECALL 
pLIMLIC STDCALL lim_create_lic_fromfile(pLIMENV pEnv, const char *szKeyFile, int *pnErr);

PRECALL 
pLIMLIC STDCALL lim_create_lic_fromkey(pLIMENV pEnv, const char *szKey, int *pnErr);

PRECALL 
pLIMLIC STDCALL lim_create_lic(pLIMENV pEnv, int *pnErr);

PRECALL 
int STDCALL lim_write_lickey(pLIMLIC pLic, const char *szKeyFile);

PRECALL 
int STDCALL lim_write_lickey_carr(pLIMLIC pLic, const char *szKeyFile);

PRECALL 
int STDCALL lim_write_secretmsg(pLIMENV pEnv, const char *szPlainText, const char *szSecretMsgFile);

PRECALL 
char* STDCALL lim_get_lickey(pLIMLIC pLic);

PRECALL 
int STDCALL lim_fget_lickey(const char *szKeyFile, char *szKey);

PRECALL
int STDCALL lim_exd_bstream(pLIMENV pEnv, char *pachBufferIn, int nBytesIn, char *pachBufferOut, int *pnBytesOut, int nDirection);

PRECALL 
int STDCALL lim_get_keypair(pLIMENV pEnv, char *szP, char *szQ, char *szN);

PRECALL 
int STDCALL lim_new_keypair(pLIMENV pEnv, int nKeySize);

PRECALL 
int STDCALL lim_gen_keypair(int nKeySize, int nSeed, const char *szProductCode);

PRECALL 
int STDCALL lim_gen_keypair_ecp(int nKeySize, int nSeed);

PRECALL 
int STDCALL lim_write_keypair(pLIMENV pEnv, const char *szPrivateKeyFile, const char *szPublicKeyFile);

PRECALL 
int STDCALL lim_write_keypair_carr(pLIMENV pEnv, const char *szPrivateKeyFile, const char *szPublicKeyFile);

PRECALL 
int STDCALL lim_read_public_key(pLIMENV pEnv, const char *szPublicKeyFile);

PRECALL 
int STDCALL lim_read_private_key(pLIMENV pEnv, const char *szPrivateKeyFile);

PRECALL 
int STDCALL lim_hash_file(const char *szKeyFile, char *szHash, int nHashSize);

PRECALL 
int STDCALL lim_load_public_key(pLIMENV pEnv, const char *szPublicKey);

PRECALL 
int STDCALL lim_load_private_key(pLIMENV pEnv, const char *szPrivateKeyP, const char *szPrivateKeyQ);

PRECALL 
int STDCALL lim_set_rsa_dir(pLIMENV pEnv, int nFlag);

PRECALL 
int STDCALL lim_parse_keypair(pLIMENV pEnv);

PRECALL 
int STDCALL lim_keys_loaded(pLIMENV pEnv);

PRECALL 
int STDCALL lim_copy_lic2str(pLIMLIC pLic, char *szKey);

PRECALL 
int STDCALL lim_copy_str2lic(const char *szBuffer, pLIMLIC pLic);

PRECALL
int STDCALL lim_print_lic(pLIMLIC pLic, char *szBuf);

PRECALL 
char* STDCALL lim_base16(unsigned char *pachBufferIn, int nBytesIn);

PRECALL
int STDCALL lim_get_hostid(char *szHostId, int nHostIdMask);

PRECALL
int STDCALL lim_get_macid(char *szMacId, int nId);

PRECALL
int STDCALL lim_get_cpuid(char *szCpuId);

PRECALL
int STDCALL lim_get_userid(char *szUserId);

PRECALL
int STDCALL lim_get_dmiuuid(char *szDMI);

PRECALL
char* STDCALL lim_canonical_uuid(const char *szUUID);

PRECALL
int STDCALL lim_gen_rkey(pLIMENV pEnv, int nSeed, int nLen, int nBase, char *szRandKey);

PRECALL 
int STDCALL lim_gen_uuid(pLIMENV pEnv, int nSeed, char *szRandUUID);

PRECALL 
int STDCALL lim_gen_product_id(pLIMENV pEnv, int nSeed, char *szRandProductId);

PRECALL 
int STDCALL lim_get_hash16(const char *pachBufferIn, int nBytesIn, char *szBuf);

PRECALL 
int STDCALL lim_get_hash12(const char *pachBufferIn, int nBytesIn, char *szBuf);

PRECALL 
int STDCALL  lim_get_hash8(const char *pachBufferIn, int nBytesIn, char *szBuf);

PRECALL
int STDCALL lim_get_sha(const char *pachBufferIn, int nBytesIn, char *pachHash, int nHashBits);

PRECALL 
int STDCALL lim_get_licsha(pLIMLIC pLic, char *pachHash);

PRECALL
int STDCALL lim_get_verkey(int *pnMajor, int *pnMinor, int *pnRelease, int *pnRevision, char *szBuild);

PRECALL
int STDCALL lim_get_verstr(char *szVer);

/* error functions */
PRECALL
void STDCALL lim_set_lasterr(pLIMENV pEnv, int nErr);

PRECALL
const char* STDCALL lim_errctx(pLIMENV pEnv);

PRECALL 
const char* STDCALL lim_errmacro(pLIMENV pEnv, int  nErrCode);

PRECALL
const char* STDCALL lim_errmsg(pLIMENV pEnv, int  nErrCode);

PRECALL 
int STDCALL lim_get_errmsg(pLIMENV pEnv, int  nErrCode, char *szBuf);

/* license properties */
PRECALL 
int STDCALL lim_platform_id(pLIMLIC pLic);

PRECALL 
const char* STDCALL lim_product_id(pLIMLIC pLic);

PRECALL 
int STDCALL lim_days_to_expiry(pLIMLIC pLic);

PRECALL 
int STDCALL lim_major_version(pLIMLIC pLic);

PRECALL 
int STDCALL lim_minor_version(pLIMLIC pLic);

PRECALL 
int STDCALL lim_type_id(pLIMLIC pLic);

PRECALL 
int STDCALL lim_size_id(pLIMLIC pLic);

PRECALL 
int STDCALL lim_has_feature(pLIMLIC pLic, int idSlot, int nMask);

PRECALL 
char* STDCALL lim_lic_id(pLIMLIC pLic);

/* license validate */
PRECALL 
int STDCALL lim_is_verified(pLIMLIC pLic);

PRECALL 
int STDCALL lim_assert_valid(pLIMLIC pLic);

PRECALL 
int STDCALL lim_sign_lic(pLIMLIC pLic);

PRECALL 
int STDCALL lim_veri_lic(pLIMLIC pLic);

/* steganography */
PRECALL 
int STDCALL lim_st_encode(char *szBMPFile, char *szSecretText, char *szBMPFileOut);

PRECALL 
char* STDCALL lim_st_decode(char *szBMPFileOut, char *szOutputFile, int *pnErr);

/* b64 */
PRECALL 
void STDCALL lim_b64_decode( char *inbuffer, int inbytes, char *outbuffer, int *outbytes);

PRECALL 
void STDCALL lim_b64_encode( char *inbuffer, int inbytes, char *outbuffer, int *outbytes);

#ifdef __cplusplus
}
#endif
#endif
