/** \example limgen.c
 * \brief
 * This is a utility application to generate and verify
 * license keys with specified options and features.
 */
#include "liman.h"
#include "getopt_long.c"

#define VERSION() {\
  int maj,min,release,revision;\
  lim_get_verkey(&maj,&min,&release,&revision,szcomment);\
  fprintf(stdout,"Liman Version %d.%d.%d.%d (%s)\n",maj,min,release,revision,szcomment);\
  }

///
#define CKERR(nErr) if (nErr!=0) goto ErrReturn

static struct option long_options[] =
{
  {"platform",    required_argument, 0, 'A',"PLATFORM_ID","Specify '%s' as the platform identifier (integer in 0..255)"},
  {"product",     required_argument, 0, 'P',"PRODUCT_CODE","Specify '%s' as the product code (string 12 chars max)"},
  {"major",       required_argument, 0, 'M',"MAJOR","Specify '%s' as the major version number (integer in 0..255)"},
  {"minor",       required_argument, 0, 'J',"MINOR","Specify '%s' as the minor version number (integer in 0..255)"},
  {"size",        required_argument, 0, 'S',"SIZE_ID","Specify '%s' as the license size identifier (integer in 0..255)"},
  {"type",        required_argument, 0, 't',"TYPE_ID","Specify '%s' as the license type identifier (integer in 0..255)"},
  {"seats",       required_argument, 0, 'o',"NSEATS","Specify '%s' as the number of seats/users allowed (integer)"},
  {"expiry",      required_argument, 0, 'e',"YYYYMMDD","Specify '%s' as the expiry date (string in YYYYMMDD format)"},
  {"serial",      required_argument, 0, 'r',"SERIAL","Specify '%s' as the serial code (string 24 chars max)"},
  {"hostid",      required_argument, 0, 'H',"HOSTID","Specify '%s' as the hostid of target machine (string 12 chars max)"},
  {"userid",      required_argument, 0, 'u',"USERID","Specify '%s' as the userid of customer (string 12 chars max)"},
  {"uuid",        required_argument, 0, 'i',"UUID","Specify '%s' as the uuid for the product (string 32 chars max)"},
  {"features",    required_argument, 0, 'f',"B1,B2,B3,B4","Specify '%s' as the bitmasks assigned to features 1,2,3 and 4, respectively"},
  {"privkeyfile", required_argument, 0, 'I',"PRIVKEYFILE","Specify '%s' as the private key file"},
  {"pubkeyfile",  required_argument, 0, 'U',"PUBKEYFILE","Specify '%s' as the public key file"},
  {"keysize",     required_argument, 0, 'k',"KEYSIZE","Specify '%s' as the key size in bits (integer)"},
  {"hashsize",    required_argument, 0, 'Z',"HASHSIZE","Specify '%s' as the hash size in bits (integer)"},
  {"seed",        required_argument, 0, 's',"SEED","Specify '%s' as the seed to initialize random number generator (integer)"},
  {"hostmask",    required_argument, 0, 'm',"HOSTMASK","Specify '%s' as the bitmask defining host identifier (1,2,4,8,16)"},
  {"licinfo",     required_argument, 0, 'L',"LICENSEFILE","Display license information"},
  {"email",       required_argument, 0, 'E',"EMAIL","Customer's email"},
  {"genkey",      no_argument,       0, 'g',NULL,"Generate keypair for this product"},
  {"genuuid",     no_argument,       0, 'D',NULL,"Generate hexadecimal uuid for this product"},
  {"testapp",     no_argument,       0, 'T',NULL,"Generate license features for 'testapp' application"},
  {"help",        no_argument,       0, 'h',NULL,"Display usage"},
  {0, 0, 0, 0, 0, 0}
};

//
//
//
void usage(void) {
  struct option *opt;
  char strbuf[255], argdesc[255];
  opt = long_options;
  fprintf(stdout,"\nUsage: limgen [options]\n");
  while (opt->name) {
    if (opt->has_arg!=no_argument) {
      sprintf(argdesc,opt->descr,opt->argval);
      sprintf(strbuf,"--%s=%s",opt->name,opt->argval);
    } else {
      sprintf(strbuf,"--%s",opt->name);
      strcpy(argdesc,opt->descr);
    }
    fprintf(stdout,"\n    -%c, %-22s  %s",opt->val,strbuf,argdesc);
    opt++;
  }

  fprintf(stdout,"\n\n"
    "Example:\n"
    "\n"
    "\n  $ limgen --product=PRODUCT1 --major=2 --minor=1 --serial=123456 \n"
    "       --expiry=20140401 --features=37,3,3,5"
    "\n\n"
    "    Generates a license with product-id 'PRODUCT1' for version '2.1' with\n"
    "    serial number '123456' with an expiration date 'Apr 1, 2014' with feature\n"
    "    masks 37,3,3 and 5 for slots 0,1,2 and 3 respectively\n"
    "\n\n"
    "    $ limgen --platform=8 --product=PRODUCT1 --major=2 --minor=1 --serial=123456\n"
    "       --privkeyfile=/path/to/priv/key --pubkeyfile=/path/to/pub/key"
    "\n\n"
    "    Generates a license with product-id 'PRODUCT1' for version '2.1' with\n"
    "    serial number '123456' for platform '8' using the specified keypair."
    "\n"
    "\n"
    );

  return;
}

//! [All TestApp Features]
//
// Set up features for 'apps/c/testapp' sample project
//
void set_testapp_features(char feature[4]) {

  feature[0]=0; //initialize all bits to zero
  feature[0] |= LIMOPT_FEATURE_0;    //turn on bit 0
  feature[0] |= LIMOPT_FEATURE_2;    //turn on bit 2
  feature[0] |= LIMOPT_FEATURE_5;    //turn on bit 5

  feature[1]=0; //initialize all bits to zero
  feature[1] |= LIMOPT_FEATURE_0;    //turn on bit 0
  feature[1] |= LIMOPT_FEATURE_1;    //turn on bit 1
  feature[1] |= LIMOPT_FEATURE_5;    //turn on bit 5

  feature[2]=0; //initialize all bits to zero
  feature[2] |= LIMOPT_FEATURE_6;    //turn on bit 6
  feature[2] |= LIMOPT_FEATURE_7;    //turn on bit 7

  feature[3]=0; //initialize all bits to zero
  feature[3] |= LIMOPT_FEATURE_0;    //turn on bit 0
  feature[3] |= LIMOPT_FEATURE_7;    //turn on bit 7

}
//! [All TestApp Features]

//
//
//
int main (int argc, char **argv)
{
 int  nErr, nout=0,outbytes=0, c, option_index;
 char strbuf[LIMPAR_MAX_LIC_STRING_LEN];
 /* data */
 pLIMENV pEnv=NULL;
 pLIMLIC pLic=NULL;
 int  mPlatformId=LIMPLAT_WIN32X86;
 char szProductId[LIMPAR_MAX_LEN_PRODUCTID+1]="LIMPROD_A1";
 int  nMajorVersion=1;
 int  nMinorVersion=0;
 int  mSizeId=LIMSIZE_STANDARD;
 int  mTypeId=LIMTYPE_DEVELOPMENT;
 int  nUsers=1, nHostIdMask=0;
 int  nExpiration=LIMPAR_NONE;
 char szSerial[LIMPAR_MAX_LEN_SERIAL+1]="";
 char uuid[2*LIMPAR_MAX_LEN_UUID+1]="";
 char szHostid[LIMPAR_MAX_LEN_HOSTID+1],*ptrHostid=NULL;
 char szUserid[LIMPAR_MAX_LEN_USERID+1],*ptrUserid=NULL;
 char features[LIMPAR_NUM_FEATURE_SLOTS] = { 0, 0, 0, 0};
 char szPrivateKeyFile[LIMPAR_MAX_PATH]="usr_private.key";
 char szPublicKeyFile[LIMPAR_MAX_PATH]="usr_public.key";
 char szLicenseKeyFile[LIMPAR_MAX_PATH]="";
 char szEmail[255]="";
 const char *szPublicKey=NULL;
 int  keysize=-1;
 int  hashsize=-1;
 int  seed=-1;
 int  isGenKeys=0, nKeyLength=0, isGenUUID=0, isLicInfo=0;

 lim_get_verstr(strbuf);
 fprintf(stdout,"\nLiman Version %s\n",strbuf);
 fflush(stdout);

 if (argc==1) {
   usage();
   exit(0);
 }
 // default host is current host
 nErr = lim_get_hostid(szHostid,0);
#if 0
 if (nErr != LIM_OK) goto ErrReturn;
 fprintf(stdout,"\nHostid %s\n",szHostid);
#endif
 
 while (1)
 {
   c = getopt_long (argc, argv, "Th?A:P:M:J:S:t:u:e:r:h:if:I:U:k:Z:Dgs:m:L:o:E:H:",
     long_options, &option_index);

   /* Detect the end of the options. */
   if (c == -1)
     break;

   switch (c)
   {
    case 'h':
    case '?':
      /* getopt_long already printed an error message. */
      usage();
      exit(0);
    case 'k':
      keysize = atoi(optarg);
      break;
    case 'm':
      nHostIdMask = atoi(optarg);
      break;
    case 'Z':
      hashsize = atoi(optarg);
      break;
    case 's':
      seed = atoi(optarg);
      break;
    case 'A':
      mPlatformId = atoi(optarg);
      break;
    case 'P':
      strncpy(szProductId, optarg, LIMPAR_MAX_LEN_PRODUCTID+1);
      break;
    case 'M':
      nMajorVersion = atoi(optarg);
      break;
    case 'J':
      nMinorVersion = atoi(optarg);
      break;
    case 'S':
      mSizeId = atoi(optarg);
      break;
    case 't':
      mTypeId = atoi(optarg);
      break;
    case 'T':
      set_testapp_features(features);
      break;
    case 'o':
      nUsers = atoi(optarg);
      break;
    case 'e':
      nExpiration = atoi(optarg);
      break;
    case 'r':
      strcpy(szSerial,optarg);
      break;
    case 'H':
      strcpy(szHostid,optarg);
      ptrHostid=szHostid;
      break;
    case 'u':
      strcpy(szUserid,optarg);
      ptrUserid=szUserid;
      break;
    case 'i':
      strcpy(uuid,optarg);
      break;
    case 'I':
      strcpy(szPrivateKeyFile,optarg);
      break;
    case 'U':
      strcpy(szPublicKeyFile,optarg);
      break;
    case 'f':
      {
        char seps[]=",", *ptr=NULL, strbuf[255], *token;
        int k=0;
        strcpy(strbuf,optarg);
        ptr=&strbuf[0];
        token=strtok(ptr,seps);
        while( token != NULL && k<LIMPAR_NUM_FEATURE_SLOTS)
        {
          features[k]=(char)atoi(token);
          k++;
          token = strtok( NULL, seps ); // C4996
        }
      }
      break;
    case 'g':
      isGenKeys=1;
      break;
    case 'D':
      isGenUUID=1;
      break;

    case 'L':
      strcpy(szLicenseKeyFile,optarg);
      isLicInfo=1;
      break;

    case 'E':
      strcpy(szEmail,optarg);
      break;

    default:
      break;

   }
 }//while

 if (keysize<0) keysize=2048;
 if (hashsize<0) hashsize=256;
 if (seed<0) seed=0;
 if (isGenKeys) {
    nErr = lim_gen_keypair(keysize,seed,NULL);
    if (nErr != LIM_OK) goto ErrReturn;
 }

 // create a new liman env
 pEnv = lim_create_env(keysize,hashsize,&nErr);
 if (nErr != LIM_OK) goto ErrReturn;

 // read private key associated with the product to sign key
 nErr = lim_read_private_key(pEnv,szPrivateKeyFile);
 CKERR(nErr);

 if (isLicInfo) {
   // read public key to decrypt message
   nErr = lim_read_public_key(pEnv,szPublicKeyFile);
   if (nErr != LIM_OK) goto ErrReturn;
   // read back the license key and verify
   pLic = lim_create_lic_fromfile(pEnv,szLicenseKeyFile,&nErr);
   if (nErr != LIM_OK) goto ErrReturn;
   // print license out
   nErr = lim_print_lic(pLic,NULL);
   if (nErr != LIM_OK) goto ErrReturn;

   nErr = lim_write_lickey_carr(pLic,NULL);
   goto ErrReturn;
 }


 // create a blank license
 pLic = lim_create_lic(pEnv,&nErr);
 if (nErr != LIM_OK) goto ErrReturn;

 if (strlen(szEmail)) {
   int _seed=0;
   lim_get_hash12(szEmail,(int)strlen(szEmail),strbuf);
   fprintf(stdout,"Customer Id: %s\n",strbuf);
 }

 if (strlen(szSerial)<=0) {
   // use a random key as serial or plug-in your own serial-generation code
   lim_gen_rkey(pEnv,0,LIMPAR_MAX_LEN_SERIAL,32,szSerial);
 }

 if (isGenUUID) {
   // use a random key as uuid or plug-in your own uuid-generation code
   lim_gen_uuid(pEnv,-1,uuid);
 }

 // generate a license key
 nErr = lim_load_lic(pLic,szProductId,mPlatformId,nMajorVersion,nMinorVersion,
   mSizeId,mTypeId,nUsers,nExpiration,features,szSerial,uuid,ptrHostid,ptrUserid,nHostIdMask);
 if (nErr != LIM_OK) goto ErrReturn;

 // write/export license key to stdout
 nErr = lim_write_lickey(pLic,NULL);
 CKERR(nErr);

 // write/export license key to file
 sprintf(strbuf,"%s/usr_%s_lic.c",szProductId,lim_lic_id(pLic));
 nErr = lim_write_lickey_carr(pLic,strbuf);
 CKERR(nErr);
 fprintf(stdout,"\n");
 fprintf(stdout,"Written license key as C/C++ code to %s\n",strbuf);

 sprintf(strbuf,"%s/usr_%s.lic",szProductId,lim_lic_id(pLic));
 nErr = lim_write_lickey(pLic,strbuf);
 CKERR(nErr);
 fprintf(stdout,"Written license key to %s\n",strbuf);


 fprintf(stdout,"Destroying the license\n");
 // destroy instance
 nErr = lim_free_lic(&pLic);
 if (nErr != LIM_OK) goto ErrReturn;

 fprintf(stdout,"\n");
 fprintf(stdout,"Reloading a new instance from .lic file..\n");
 // read public key to verify key
#if READ_PUBLIC_KEY
 nErr = lim_read_public_key(pEnv,szPublicKeyFile);
#else
 nErr = lim_load_public_key(pEnv,lim_usr_public_key_n(&nKeyLength));
#endif
 CKERR(nErr);

 // read back the license key and verify
 pLic = lim_create_lic_fromfile(pEnv,strbuf,&nErr);
 CKERR(nErr);
 fprintf(stdout,"Created successfully a new instance from '%s'..\n",strbuf);

 fprintf(stdout,"Displaying license properties and features..\n");
 nErr = lim_print_lic(pLic,strbuf);
 if (nErr != LIM_OK) goto ErrReturn;
 fprintf(stdout,"%s\n",strbuf);
 
 fprintf(stdout,"Verifying license for this host..\n");
 nErr = lim_is_verified(pLic);
 CKERR(nErr);
 fprintf(stdout,"Generated license is verified!\n");
 fprintf(stdout,"Done!\n");
ErrReturn:
 if (nErr !=LIM_OK)  fprintf(stdout,"\nError %d: %s\n",nErr,lim_errmsg(pEnv,nErr));
 lim_free_lic(&pLic);
 lim_free_env(&pEnv);

 exit (nErr);
}
