@echo off
REM LIMAN SDK Fetch, Build, and Distribution Script (Windows)
REM
REM This handy script downloads a fresh copy of the LIMAN SDK package into a temporary
REM folder (or fetches from a local repository of packages), builds the SDK with your
REM official license file, and then copies the built library to a target destination.
REM (c) Baroks Inc.
REM
REM Usage: lg_fetch_build_distrib_sdk.bat [options] [platform] [output_dir]
REM   platform: win64x86 | win32x86 | mingw64 | mingw32 (default: win64x86)
REM   output_dir: where to place built SDK (default: %TEMP%\liman_licensed)
REM
REM Options:
REM   -f               Actually execute the build (default is dry-run mode)
REM   -h               Show this help message
REM   --local          Use local release directory instead of downloading from web
REM   --target-dir=DIR Copy built libliman_sdk.lib to specified directory
REM
REM Environment Variables:
REM   LICMAKER_LICENSE_FILE_C  - Path to your official license .c file (required)
REM                              This file will be copied to license\usr_liman_lic.c
REM                              and compiled into the SDK library
REM
REM Example:
REM   set LICMAKER_LICENSE_FILE_C=C:\path\to\usr_liman_lic.c
REM   lg_fetch_build_distrib_sdk.bat -f
REM   lg_fetch_build_distrib_sdk.bat -f --local
REM   lg_fetch_build_distrib_sdk.bat -f --target-dir=C:\MyApp\lib
REM

setlocal EnableDelayedExpansion

REM Save current directory and resolve script directory
set "ORIG_DIR=%CD%"
set "SCRIPT_DIR=%~dp0"

REM Show usage if no arguments
if "%~1"=="" goto :show_usage

REM Set default license file if not specified
if "%LICMAKER_LICENSE_FILE_C%"=="" set "LICMAKER_LICENSE_FILE_C=%SCRIPT_DIR%..\license\usr_liman_lic.c"

REM Read version from liman.h if available
set "LIMAN_H=%SCRIPT_DIR%..\include\liman.h"
set "LIMVER_MAJOR=1"
set "LIMVER_MINOR=3"

if exist "%LIMAN_H%" (
    for /f "tokens=3" %%a in ('findstr /R /C:"^#define LIMVER_MAJOR" "%LIMAN_H%"') do set "LIMVER_MAJOR=%%a"
    for /f "tokens=3" %%a in ('findstr /R /C:"^#define LIMVER_MINOR" "%LIMAN_H%"') do set "LIMVER_MINOR=%%a"
)
set "LIMAN_VERSION=%LIMVER_MAJOR%.%LIMVER_MINOR%"

set "BASE_URL=https://baroks.com/Downloads"
set "LOCAL_RELEASE_DIR=%USERPROFILE%\dev\c\liman\liman-release\release"

REM Default values
set "LIMAN_PLATFORM=win64x86"
set "OUTPUT_DIR=%TEMP%\liman_licensed"
set "USE_LOCAL=0"
set "TARGET_DIR="
set "FORCE=0"

REM Parse arguments
:parse_args
if "%~1"=="" goto :done_args
if /i "%~1"=="-h" goto :show_usage
if /i "%~1"=="--help" goto :show_usage
if /i "%~1"=="-f" (
    set "FORCE=1"
    shift
    goto :parse_args
)
if /i "%~1"=="--force" (
    set "FORCE=1"
    shift
    goto :parse_args
)
if /i "%~1"=="--local" (
    set "USE_LOCAL=1"
    shift
    goto :parse_args
)
if /i "%~1"=="-local" (
    set "USE_LOCAL=1"
    shift
    goto :parse_args
)
REM Handle --target-dir=VALUE (NOTE: = causes arg splitting, so check arg1=--target-dir and arg2=value)
if /i "%~1"=="--target-dir" (
    set "TARGET_DIR=%~2"
    shift
    shift
    goto :parse_args
)
REM Check for unknown options (but skip if empty)
set "_ARG=%~1"
if not "%_ARG%"=="" if "%_ARG:~0,1%"=="-" (
    echo Unknown option: %~1
    echo Use -h for help.
    exit /b 1
)
REM Positional arguments
if "%LIMAN_PLATFORM%"=="win64x86" (
    set "LIMAN_PLATFORM=%~1"
) else (
    set "OUTPUT_DIR=%~1"
)
shift
goto :parse_args
:done_args

set "PACKAGE=liman_core-%LIMAN_VERSION%-%LIMAN_PLATFORM%.tar.gz"
set "DOWNLOAD_URL=%BASE_URL%/%PACKAGE%"

echo ========================================
echo LIMAN SDK Licensed Build Script
echo ========================================
echo Platform:     %LIMAN_PLATFORM%
echo Package:      %PACKAGE%
echo Output dir:   %OUTPUT_DIR%
echo License file: %LICMAKER_LICENSE_FILE_C%
if not "%TARGET_DIR%"=="" echo Target dir:   %TARGET_DIR%
echo Force:        %FORCE%
echo ========================================
echo.

REM Check for license file
if not exist "%LICMAKER_LICENSE_FILE_C%" (
    echo ERROR: License file not found: %LICMAKER_LICENSE_FILE_C%
    echo.
    echo Set LICMAKER_LICENSE_FILE_C environment variable to your license file path.
    exit /b 1
)

REM Dry-run mode check
if "%FORCE%"=="0" (
    echo *** DRY-RUN MODE ***
    echo.
    echo This is a preview of what would be executed.
    echo Use -f to actually run the build.
    echo.
    echo Would execute:
    echo   1. Download/copy package: %PACKAGE%
    echo   2. Extract to: %OUTPUT_DIR%
    echo   3. Install license: %LICMAKER_LICENSE_FILE_C%
    echo   4. Build SDK library
    if not "%TARGET_DIR%"=="" echo   5. Copy to: %TARGET_DIR%\lib\ and %TARGET_DIR%\include\
    echo   6. Cleanup temp directory
    echo.
    echo Run with -f to execute.
    exit /b 0
)

REM Step 1: Download or copy package
echo [Step 1] Obtaining LIMAN SDK package...
if not exist "%OUTPUT_DIR%" mkdir "%OUTPUT_DIR%"
cd /d "%OUTPUT_DIR%"

set "LOCAL_PACKAGE=%LOCAL_RELEASE_DIR%\%PACKAGE%"
if "%USE_LOCAL%"=="1" (
    if exist "%LOCAL_PACKAGE%" (
        echo Copying from local release: %LOCAL_PACKAGE%
        copy "%LOCAL_PACKAGE%" "%PACKAGE%" >nul
    ) else (
        echo ERROR: Local package not found: %LOCAL_PACKAGE%
        exit /b 1
    )
) else (
    echo Downloading from %DOWNLOAD_URL%...
    where curl >nul 2>&1
    if !errorlevel! equ 0 (
        curl -L -o "%PACKAGE%" "%DOWNLOAD_URL%"
    ) else (
        echo ERROR: curl not found. Please install curl or copy package manually.
        exit /b 1
    )
)

if not exist "%PACKAGE%" (
    echo ERROR: Package not found!
    exit /b 1
)
echo Package ready: %PACKAGE%
echo.

REM Step 2: Extract package
echo [Step 2] Extracting package...
tar -xzf "%PACKAGE%"
set "EXTRACT_DIR=liman_core-%LIMAN_VERSION%-%LIMAN_PLATFORM%"

if not exist "%EXTRACT_DIR%" (
    echo ERROR: Extracted directory %EXTRACT_DIR% not found!
    exit /b 1
)
echo Extracted to: %EXTRACT_DIR%
echo.

REM Step 3: Copy license file
echo [Step 3] Installing license file...
echo Working directory: %CD%
set "LICENSE_DEST=%EXTRACT_DIR%\license\usr_liman_lic.c"
copy "%LICMAKER_LICENSE_FILE_C%" "%LICENSE_DEST%" >nul
echo Copied to %LICENSE_DEST%

REM Also create the _copy.c file that some Makefiles expect
copy "%LICMAKER_LICENSE_FILE_C%" "%EXTRACT_DIR%\license\usr_liman_lic_copy.c" >nul
echo License installed.
echo.

REM Step 4: Navigate to lib\<platform> and build
echo [Step 4] Building libliman_sdk...
cd /d "%EXTRACT_DIR%\lib\%LIMAN_PLATFORM%"
echo Current directory: %CD%

REM Determine build tool based on platform
if "%LIMAN_PLATFORM%"=="win64x86" goto :nmake_build
if "%LIMAN_PLATFORM%"=="win32x86" goto :nmake_build
if "%LIMAN_PLATFORM:~0,5%"=="mingw" goto :mingw_build

:nmake_build
REM Setup Visual Studio environment
set "VCVARS="
if exist "C:\Program Files\Microsoft Visual Studio\2022\Community\VC\Auxiliary\Build\vcvars64.bat" (
    set "VCVARS=C:\Program Files\Microsoft Visual Studio\2022\Community\VC\Auxiliary\Build\vcvars64.bat"
)
if exist "C:\Program Files\Microsoft Visual Studio\2022\Professional\VC\Auxiliary\Build\vcvars64.bat" (
    set "VCVARS=C:\Program Files\Microsoft Visual Studio\2022\Professional\VC\Auxiliary\Build\vcvars64.bat"
)
if exist "C:\Program Files\Microsoft Visual Studio\2022\Enterprise\VC\Auxiliary\Build\vcvars64.bat" (
    set "VCVARS=C:\Program Files\Microsoft Visual Studio\2022\Enterprise\VC\Auxiliary\Build\vcvars64.bat"
)
if exist "C:\Program Files (x86)\Microsoft Visual Studio\2019\Community\VC\Auxiliary\Build\vcvars64.bat" (
    set "VCVARS=C:\Program Files (x86)\Microsoft Visual Studio\2019\Community\VC\Auxiliary\Build\vcvars64.bat"
)

if "%VCVARS%"=="" (
    echo ERROR: Visual Studio not found. Please install Visual Studio with C++ support.
    exit /b 1
)

echo Setting up Visual Studio environment...
call "%VCVARS%" >nul

echo Running: nmake
nmake
if errorlevel 1 (
    echo ERROR: Build failed!
    exit /b 1
)
echo.
goto :verify_build

:mingw_build
echo Running: mingw32-make
mingw32-make
if errorlevel 1 (
    echo ERROR: Build failed!
    exit /b 1
)
echo.
goto :verify_build

:verify_build
REM Step 5: Verify build
echo.
echo [Step 5] Verifying build...
REM Clear LIB_FILE first, then set based on platform
set "LIB_FILE="
if "%LIMAN_PLATFORM%"=="win64x86" set "LIB_FILE=libliman_sdk.lib"
if "%LIMAN_PLATFORM%"=="win32x86" set "LIB_FILE=libliman_sdk.lib"
if "%LIMAN_PLATFORM%"=="mingw64" set "LIB_FILE=libliman_sdk.a"
if "%LIMAN_PLATFORM%"=="mingw32" set "LIB_FILE=libliman_sdk.a"
if "%LIB_FILE%"=="" set "LIB_FILE=libliman_sdk.a"

if exist "%LIB_FILE%" (
    echo SUCCESS: %LIB_FILE% built
    dir "%LIB_FILE%"
) else (
    echo ERROR: %LIB_FILE% not found!
    exit /b 1
)

echo.

REM Check for tools
if exist "limutil.exe" echo limutil.exe built successfully
if exist "limgen.exe" echo limgen.exe built successfully
echo.

REM Copy to target directory if specified
if not "%TARGET_DIR%"=="" (
    echo [Step 6] Copying library to target directory...
    if not exist "%TARGET_DIR%\lib" mkdir "%TARGET_DIR%\lib"
    copy "%LIB_FILE%" "%TARGET_DIR%\lib\" >nul
    echo Library copied to: %TARGET_DIR%\lib\%LIB_FILE%
    REM Also copy liman.h to include directory
    if not exist "%TARGET_DIR%\include" mkdir "%TARGET_DIR%\include"
    copy "%LIMAN_H%" "%TARGET_DIR%\include\" >nul
    echo Header copied to: %TARGET_DIR%\include\liman.h
    echo.
)

REM Summary
echo ========================================
echo Licensed SDK Build Complete!
echo ========================================
echo Platform:      %LIMAN_PLATFORM%
if not "%TARGET_DIR%"=="" (
    echo Library:       %TARGET_DIR%\lib\%LIB_FILE%
    echo Header:        %TARGET_DIR%\include\liman.h
)
echo ========================================

REM Cleanup temp directory
echo Cleaning up temp directory: %OUTPUT_DIR%
cd /d "%ORIG_DIR%"
rmdir /s /q "%OUTPUT_DIR%" 2>nul

endlocal
goto :eof

:show_usage
echo Usage: %~nx0 [options] [platform] [output_dir]
echo.
echo Downloads LIMAN SDK, builds it with your license file, and optionally
echo copies the built library to a target directory.
echo.
echo Arguments:
echo   platform        Platform: win64x86 ^| win32x86 ^| mingw64 ^| mingw32
echo                   (default: win64x86)
echo   output_dir      Temporary build directory (default: %%TEMP%%\liman_licensed)
echo.
echo Options:
echo   -f               Actually execute the build (default is dry-run mode)
echo   -h               Show this help message
echo   --local          Use local release directory instead of downloading
echo   --target-dir=DIR Copy built library and headers to specified directory
echo.
echo Environment Variables:
echo   LICMAKER_LICENSE_FILE_C  Path to your license .c file
echo                            (default: %SCRIPT_DIR%..\license\usr_liman_lic.c)
echo.
echo Examples:
echo   %~nx0 -f
echo   %~nx0 -f --local
echo   %~nx0 -f --target-dir=C:\MyApp
echo   set LICMAKER_LICENSE_FILE_C=C:\path\to\lic.c ^& %~nx0 -f win64x86
exit /b 0
